import { E as EventEmitter, i as injectTestingHelpers, c as createContextManager, N as NatsAuthService, R as RPCManager, B as Browser } from './Browser.service-DK0W0fjf.js';
import './ContextDetector-DxzeZq05.js';

class Tabs {
  constructor() {
    this.name = "Tabs";
    this.eventEmitter = new EventEmitter();
  }
  async initialize() {
    this.setupChromeEventListeners();
  }
  async teardown() {
    this.eventEmitter.removeAllListeners();
  }
  setupChromeEventListeners() {
    chrome.tabs.onCreated.addListener((tab) => {
      this.eventEmitter.emit("tabCreated", this.convertChromeTab(tab));
    });
    chrome.tabs.onUpdated.addListener((_, __, tab) => {
      this.eventEmitter.emit("tabUpdated", this.convertChromeTab(tab));
    });
    chrome.tabs.onRemoved.addListener((tabId) => {
      this.eventEmitter.emit("tabRemoved", { id: tabId.toString() });
    });
    chrome.tabs.onActivated.addListener(async (activeInfo) => {
      const tab = await chrome.tabs.get(activeInfo.tabId);
      this.eventEmitter.emit("tabUpdated", this.convertChromeTab(tab));
    });
  }
  convertChromeTab(chromeTab) {
    return {
      id: chromeTab.id?.toString() || "",
      url: chromeTab.url || "",
      title: chromeTab.title || "",
      active: chromeTab.active || false
    };
  }
  async getTabs() {
    try {
      const chromeTabs = await chrome.tabs.query({});
      return chromeTabs.map((tab) => this.convertChromeTab(tab));
    } catch (error) {
      console.error("Tabs.getTabs error:", error);
      this.emitTabError("query", error);
      return [];
    }
  }
  async getTab({ id }) {
    try {
      const chromeTab = await chrome.tabs.get(parseInt(id, 10));
      return this.convertChromeTab(chromeTab);
    } catch (error) {
      this.emitTabError(id, error);
      throw new Error(`No tab with id: ${id}`);
    }
  }
  async createTab(tabData) {
    try {
      const chromeTab = await chrome.tabs.create({
        url: tabData.url,
        active: tabData.active
      });
      if (!chromeTab.id) {
        throw new Error("Tab creation failed");
      }
      for (let i = 0; i < 10; i++) {
        const tab = await chrome.tabs.get(chromeTab.id);
        if (tab.url) {
          return this.convertChromeTab(tab);
        }
        await new Promise((resolve) => setTimeout(resolve, 100));
      }
      throw new Error("Tab creation failed");
    } catch (error) {
      this.emitTabError("create", error);
      throw error;
    }
  }
  async updateTab(updateData) {
    try {
      const chromeTab = await chrome.tabs.update(parseInt(updateData.id, 10), {
        url: updateData.data.url,
        active: updateData.data.active
      });
      if (!chromeTab.id) {
        throw new Error(`Failed to update tab ${updateData.id}: updated tab has no id`);
      }
      const tab = await chrome.tabs.get(chromeTab.id);
      if (!tab) {
        throw new Error(`Failed to get updated tab ${chromeTab.id}`);
      }
      await new Promise((resolve) => setTimeout(resolve, 500));
      if (updateData.data.active !== void 0 && tab.active !== updateData.data.active) {
        throw new Error(`Failed to update tab ${chromeTab.id} active state`);
      }
      const finalTab = await chrome.tabs.get(chromeTab.id);
      return this.convertChromeTab(finalTab);
    } catch (error) {
      this.emitTabError(updateData.id, error);
      throw error;
    }
  }
  async closeTab({ id }) {
    try {
      await chrome.tabs.remove(parseInt(id, 10));
      return true;
    } catch (error) {
      this.emitTabError(id, error);
      throw new Error(`Failed to close tab ${id}: ${error.message}`);
    }
  }
  emitTabError(tabId, error) {
    this.eventEmitter.emit("tabError", tabId, error);
  }
}

class Session {
  constructor() {
    this.name = "Session";
    this.eventEmitter = new EventEmitter();
  }
  async initialize() {
  }
  async teardown() {
    this.eventEmitter.removeAllListeners();
  }
  convertSameSiteValue({ value }) {
    if (!value) return undefined;
    return value.charAt(0).toUpperCase() + value.slice(1);
  }
  convertSameSiteValueToChrome({ value }) {
    if (!value) return undefined;
    return value.toLowerCase();
  }
  convertChromeCookie({ chromeCookie }) {
    return {
      name: chromeCookie.name,
      value: chromeCookie.value,
      domain: chromeCookie.domain,
      path: chromeCookie.path,
      secure: chromeCookie.secure,
      httpOnly: chromeCookie.httpOnly,
      sameSite: this.convertSameSiteValue({ value: chromeCookie.sameSite }),
      expirationDate: chromeCookie.expirationDate,
      storeId: chromeCookie.storeId,
      session: chromeCookie.session,
      hostOnly: chromeCookie.hostOnly
    };
  }
  async getCookies({ domain }) {
    try {
      const chromeCookies = await chrome.cookies.getAll({ domain });
      return chromeCookies.map((cookie) => this.convertChromeCookie({ chromeCookie: cookie }));
    } catch (error) {
      this.emitSessionError({ operation: "getCookies", error });
      throw error;
    }
  }
  async getCookie({ domain, name }) {
    try {
      const cookies = await this.getCookies({ domain });
      return cookies.find((cookie) => cookie.name === name);
    } catch (error) {
      this.emitSessionError({ operation: "getCookie", error });
      return undefined;
    }
  }
  async setCookie({ domain, cookie }) {
    try {
      const chromeCookie = await chrome.cookies.set({
        url: `https://${domain}${cookie.path || "/"}`,
        name: cookie.name,
        value: cookie.value,
        domain: cookie.domain || `.${domain}`,
        path: cookie.path || "/",
        secure: cookie.secure,
        httpOnly: cookie.httpOnly,
        sameSite: this.convertSameSiteValueToChrome({ value: cookie.sameSite }),
        expirationDate: cookie.expirationDate
      });
      return chromeCookie ? this.convertChromeCookie({ chromeCookie }) : void 0;
    } catch (error) {
      this.emitSessionError({ operation: "setCookie", error });
      throw error;
    }
  }
  async removeCookie({ domain, name }) {
    try {
      const cookie = await this.getCookie({ domain, name });
      if (!cookie) {
        return false;
      }
      const result = await chrome.cookies.remove({
        url: `https://${domain}${cookie.path}`,
        name: cookie.name
      });
      return result !== null;
    } catch (error) {
      this.emitSessionError({ operation: "removeCookie", error });
      return false;
    }
  }
  emitSessionError({ operation, error }) {
    this.eventEmitter.emit("error", operation, error);
  }
}

class Page {
  constructor() {
    this.name = "Page";
    this.eventEmitter = new EventEmitter();
  }
  async initialize() {
  }
  async teardown() {
    this.eventEmitter.removeAllListeners();
  }
  async executeInPage({ tabId, script }) {
    try {
      const scriptStr = typeof script === "function" ? `(${script.toString()})()` : script;
      const result = await chrome.scripting.executeScript({
        target: { tabId },
        func: function(scriptStr2) {
          return new Promise(async (resolve) => {
            try {
              const fn = new Function(`return (async () => { ${scriptStr2} })();`);
              const result2 = await fn();
              resolve(result2);
            } catch (error) {
              resolve({
                __error: true,
                message: error.message,
                stack: error.stack
              });
            }
          });
        },
        args: [scriptStr],
        world: "ISOLATED"
      });
      if (result[0]?.result?.__error) {
        throw new Error(`Script execution failed: ${result[0].result.message}`);
      }
      return result[0].result;
    } catch (error) {
      this.emitPageError("executeInPage", error);
      throw error;
    }
  }
  async executeDOMOperation({ tabId, operation }) {
    return this.executeInPage({ tabId, script: operation });
  }
  async observe({ tabId, selector }) {
    const observerId = Math.random().toString(36).substring(7);
    await this.executeInPage({
      tabId,
      script: `
                // Initialize storage if needed
                window._mutationStore = window._mutationStore || {};
                window._mutationStore['${observerId}'] = [];
                
                // console.log('[PageService:Page] Setting up mutation observer...');
                const element = document.querySelector('${selector}');
                if (!element) {
                    // console.error('[PageService:Page] Element not found');
                    throw new Error('Element not found');
                }

                // Create a promise that resolves when mutations are detected
                window._mutationPromises = window._mutationPromises || {};
                window._mutationPromises['${observerId}'] = [];
                
                const observer = new MutationObserver((mutations) => {
                    // console.log('[PageService:Page] Mutations detected:', mutations.length);
                    const records = mutations.map(record => ({
                        type: record.type,
                        target: {
                            tagName: record.target.tagName,
                            id: record.target.id,
                            className: record.target.className
                        },
                        addedNodes: Array.from(record.addedNodes).map(node => ({
                            tagName: node.nodeName
                        })),
                        removedNodes: Array.from(record.removedNodes).map(node => ({
                            tagName: node.nodeName
                        })),
                        attributeName: record.attributeName,
                        oldValue: record.oldValue
                    }));
                    
                    window._mutationStore['${observerId}'].push(...records);
                    
                    // Resolve any pending promises and clear them
                    const promises = window._mutationPromises['${observerId}'];
                    window._mutationPromises['${observerId}'] = [];
                    promises.forEach(resolve => resolve(records));
                });

                // Observe all types of mutations
                observer.observe(element, {
                    attributes: true,
                    childList: true,
                    characterData: true,
                    subtree: true
                });
                
                // Store observer reference
                window._observers = window._observers || {};
                window._observers['${observerId}'] = observer;
                
                // Return the observer ID
                return observerId;
            `
    }).catch((error) => {
      console.error(`[PageService] Error setting up observer:`, error);
      this.emitPageError("observe", error);
      throw error;
    });
    const pollInterval = setInterval(async () => {
      try {
        const records = await this.executeInPage({
          tabId,
          script: `
                        return new Promise((resolve) => {
                            const records = window._mutationStore['${observerId}'] || [];
                            if (records.length > 0) {
                                window._mutationStore['${observerId}'] = []; // Clear the store
                                resolve(records);
                            } else {
                                // Store the resolve function to be called when mutations occur
                                // Only store if we don't have too many pending promises
                                if (window._mutationPromises['${observerId}'].length < 10) {
                                    window._mutationPromises['${observerId}'].push(resolve);
                                    // Set a timeout to resolve with empty array if no mutations occur
                                    setTimeout(() => {
                                        const index = window._mutationPromises['${observerId}'].indexOf(resolve);
                                        if (index !== -1) {
                                            window._mutationPromises['${observerId}'].splice(index, 1);
                                            resolve([]);
                                        }
                                    }, 500); // Increased timeout to reduce polling frequency
                                } else {
                                    resolve([]); // Resolve immediately if too many pending promises
                                }
                            }
                        });
                    `
        });
        if (records && records.length > 0) {
          this.eventEmitter.emit("mutations", { records, tabId, observerId });
        }
      } catch (error) {
        this.emitPageError("observe", error);
      }
    }, 250);
    return {
      observerId,
      unsubscribe: () => {
        clearInterval(pollInterval);
        this.executeInPage({
          tabId,
          script: `
                        if (window._observers && window._observers['${observerId}']) {
                            window._observers['${observerId}'].disconnect();
                            delete window._observers['${observerId}'];
                            delete window._mutationStore['${observerId}'];
                            // Resolve and clear any pending promises
                            const promises = window._mutationPromises['${observerId}'] || [];
                            promises.forEach(resolve => resolve([]));
                            delete window._mutationPromises['${observerId}'];
                        }
                    `
        }).catch((error) => {
          console.error(`[PageService] Error cleaning up observer:`, error);
        });
      }
    };
  }
  async evaluate({ tabId, script }) {
    const scriptStr = typeof script === "function" ? `return (${script.toString()})()` : script;
    return this.executeInPage({ tabId, script: scriptStr });
  }
  async waitForElement({ tabId, selector, options }) {
    const script = `
            return new Promise((resolve) => {
                const timeout = ${options.timeout || 5e3};
                const startTime = Date.now();
                const state = '${options.state || "visible"}';
                
                function checkElement() {
                    const element = document.querySelector('${selector}');
                    if (!element) {
                        if (state === 'detached') {
                            resolve({ textContent: null, isVisible: false });
                            return;
                        }
                        
                        if (Date.now() - startTime < timeout) {
                            setTimeout(checkElement, 100);
                        } else {
                            resolve(null);
                        }
                        return;
                    }
                    
                    const rect = element.getBoundingClientRect();
                    const isVisible = rect.width > 0 && rect.height > 0;
                    
                    if (
                        (state === 'visible' && isVisible) ||
                        (state === 'hidden' && !isVisible) ||
                        state === 'attached'
                    ) {
                        resolve({
                            textContent: element.textContent?.trim() || null,
                            isVisible
                        });
                        return;
                    }
                    
                    if (Date.now() - startTime < timeout) {
                        setTimeout(checkElement, 100);
                    } else {
                        resolve(null);
                    }
                }
                
                checkElement();
            });
        `;
    return this.executeDOMOperation({ tabId, operation: script });
  }
  async waitForNavigation({ tabId, condition }) {
    try {
      await new Promise((resolve, reject) => {
        let timeoutId;
        let isResolved = false;
        const cleanup = () => {
          chrome.tabs.onUpdated.removeListener(listener);
          if (timeoutId) clearTimeout(timeoutId);
        };
        const listener = (_tabId, changeInfo) => {
          if (_tabId === tabId && !isResolved) {
            switch (condition) {
              case "load":
                if (changeInfo.status === "complete") {
                  isResolved = true;
                  cleanup();
                  resolve();
                }
                break;
              case "domcontentloaded":
                if (changeInfo.status === "loading") {
                  isResolved = true;
                  cleanup();
                  resolve();
                }
                break;
              case "change":
                isResolved = true;
                cleanup();
                resolve();
                break;
              case "networkidle0":
              case "networkidle2":
                if (changeInfo.status === "complete") {
                  const timer = setTimeout(() => {
                    isResolved = true;
                    cleanup();
                    resolve();
                  }, condition === "networkidle0" ? 500 : 2e3);
                  timeoutId = timer;
                }
                break;
            }
          }
        };
        chrome.tabs.onUpdated.addListener(listener);
        timeoutId = setTimeout(() => {
          if (!isResolved) {
            cleanup();
            reject(new Error("Navigation timeout"));
          }
        }, 3e4);
      });
    } catch (error) {
      this.emitPageError("waitForNavigation", error);
      throw error;
    }
  }
  async navigate({ tabId, url }) {
    try {
      await chrome.tabs.update(tabId, { url });
    } catch (error) {
      this.emitPageError("navigate", error);
      throw error;
    }
  }
  async back({ tabId }) {
    try {
      await chrome.tabs.goBack(tabId);
    } catch (error) {
      this.emitPageError("back", error);
      throw error;
    }
  }
  async forward({ tabId }) {
    try {
      await chrome.tabs.goForward(tabId);
    } catch (error) {
      this.emitPageError("forward", error);
      throw error;
    }
  }
  async reload({ tabId }) {
    try {
      await chrome.tabs.reload(tabId);
    } catch (error) {
      this.emitPageError("reload", error);
      throw error;
    }
  }
  emitPageError(operation, error) {
    console.error(`[PageService] Error in ${operation}:`, error);
    this.eventEmitter.emit("error", operation, error);
  }
  // Helper method to handle message-based interactions
  async executeInteraction(tabId, type, payload) {
    const id = Math.random().toString(36).substring(7);
    const responsePromise = new Promise((resolve) => {
      const handler = (message, sender) => {
        if (sender.tab?.id !== tabId) return;
        if (message.type === `${type}_RESPONSE` && message.id === id) {
          chrome.runtime.onMessage.removeListener(handler);
          resolve(message.payload);
        }
      };
      chrome.runtime.onMessage.addListener(handler);
    });
    await chrome.tabs.sendMessage(tabId, {
      type: `${type}_REQUEST`,
      payload: {
        id,
        args: payload
      }
    });
    return responsePromise;
  }
  // Extraction.service
  async extract({ tabId, config }) {
    return this.executeInteraction(tabId, "EXTRACT", [config]);
  }
  // Selection.service
  async $({ tabId, selector, rootSelector }) {
    return this.executeInteraction(tabId, "$", [selector, rootSelector]);
  }
  async $$({ tabId, selector, rootSelector }) {
    return this.executeInteraction(tabId, "$$", [selector, rootSelector]);
  }
  async elementFromPoint({ tabId, x, y }) {
    return this.executeInteraction(tabId, "ELEMENTFROMPOINT", [x, y]);
  }
  // Interaction.service
  async moveMouse({ tabId, target }) {
    return this.executeInteraction(tabId, "MOVEMOUSE", [target]);
  }
  async click({ tabId, selector }) {
    return this.executeInteraction(tabId, "CLICK", [selector]);
  }
  async hover({ tabId, selector }) {
    return this.executeInteraction(tabId, "HOVER", [selector]);
  }
  async type({ tabId, selector, text }) {
    return this.executeInteraction(tabId, "TYPE", [selector, text]);
  }
  async focus({ tabId, selector }) {
    return this.executeInteraction(tabId, "FOCUS", [selector]);
  }
  async blur({ tabId, selector }) {
    return this.executeInteraction(tabId, "BLUR", [selector]);
  }
  async press({ tabId, key }) {
    return this.executeInteraction(tabId, "PRESS", [key]);
  }
  async drag({ tabId, sourceSelector, targetSelector }) {
    return this.executeInteraction(tabId, "DRAG", [sourceSelector, targetSelector]);
  }
  async scroll({ tabId, x, y }) {
    return this.executeInteraction(tabId, "SCROLL", [x, y]);
  }
  async scrollTo({ tabId, x, y }) {
    return this.executeInteraction(tabId, "SCROLLTO", [x, y]);
  }
  async scrollIntoView({ tabId, selector }) {
    return this.executeInteraction(tabId, "SCROLLINTOVIEW", [selector]);
  }
  async dispatchEvent({ tabId, eventName, selector, detail }) {
    return this.executeInteraction(tabId, "DISPATCHEVENT", [eventName, selector, detail]);
  }
  async setValue({ tabId, selector, value }) {
    return this.executeInteraction(tabId, "SETVALUE", [selector, value]);
  }
}

class Network {
  constructor() {
    this.name = "Network";
    this.eventEmitter = new EventEmitter();
    this.ruleIds = [];
  }
  async initialize() {
    await this.setupDeclarativeRules();
  }
  async teardown() {
    if (this.ruleIds.length) {
      await chrome.declarativeNetRequest.updateDynamicRules({
        removeRuleIds: this.ruleIds
      });
    }
  }
  async setupDeclarativeRules() {
    const existingRules = await chrome.declarativeNetRequest.getDynamicRules();
    const removeRuleIds = existingRules.map((rule) => rule.id);
    if (removeRuleIds.length) {
      await chrome.declarativeNetRequest.updateDynamicRules({
        removeRuleIds
      });
    }
    const rules = [
      // Remove problematic security headers
      {
        id: 1,
        priority: 1,
        action: {
          type: chrome.declarativeNetRequest.RuleActionType.MODIFY_HEADERS,
          responseHeaders: [
            { header: "content-security-policy", operation: chrome.declarativeNetRequest.HeaderOperation.REMOVE },
            { header: "content-security-policy-report-only", operation: chrome.declarativeNetRequest.HeaderOperation.REMOVE },
            { header: "x-frame-options", operation: chrome.declarativeNetRequest.HeaderOperation.REMOVE },
            { header: "cross-origin-opener-policy", operation: chrome.declarativeNetRequest.HeaderOperation.REMOVE }
          ]
        },
        condition: {
          urlFilter: "*",
          resourceTypes: [
            chrome.declarativeNetRequest.ResourceType.MAIN_FRAME,
            chrome.declarativeNetRequest.ResourceType.SUB_FRAME,
            chrome.declarativeNetRequest.ResourceType.SCRIPT,
            chrome.declarativeNetRequest.ResourceType.XMLHTTPREQUEST
          ]
        }
      },
      // Add required security headers
      {
        id: 2,
        priority: 1,
        action: {
          type: chrome.declarativeNetRequest.RuleActionType.MODIFY_HEADERS,
          requestHeaders: [
            { header: "sec-fetch-mode", operation: chrome.declarativeNetRequest.HeaderOperation.SET, value: "navigate" },
            { header: "sec-fetch-dest", operation: chrome.declarativeNetRequest.HeaderOperation.SET, value: "document" },
            { header: "sec-fetch-site", operation: chrome.declarativeNetRequest.HeaderOperation.SET, value: "same-origin" },
            { header: "sec-fetch-user", operation: chrome.declarativeNetRequest.HeaderOperation.SET, value: "?1" }
          ]
        },
        condition: {
          urlFilter: "*",
          resourceTypes: [
            chrome.declarativeNetRequest.ResourceType.MAIN_FRAME,
            chrome.declarativeNetRequest.ResourceType.SUB_FRAME,
            chrome.declarativeNetRequest.ResourceType.SCRIPT,
            chrome.declarativeNetRequest.ResourceType.XMLHTTPREQUEST
          ]
        }
      },
      // Add unsafe-eval to CSP
      {
        id: 3,
        priority: 1,
        action: {
          type: chrome.declarativeNetRequest.RuleActionType.MODIFY_HEADERS,
          responseHeaders: [
            {
              header: "content-security-policy",
              operation: chrome.declarativeNetRequest.HeaderOperation.APPEND,
              value: "script-src 'self' 'unsafe-inline' 'unsafe-eval' https: http: data: blob:"
            }
          ]
        },
        condition: {
          urlFilter: "*",
          resourceTypes: [
            chrome.declarativeNetRequest.ResourceType.MAIN_FRAME,
            chrome.declarativeNetRequest.ResourceType.SUB_FRAME
          ]
        }
      }
    ];
    await chrome.declarativeNetRequest.updateDynamicRules({
      addRules: rules
    });
    await chrome.declarativeNetRequest.getDynamicRules();
    this.ruleIds = rules.map((r) => r.id);
  }
}

chrome.action.onClicked.addListener(function(tab) {
  chrome.tabs.create({ url: chrome.runtime.getURL("options.html"), active: true });
});
class BackgroundService {
  constructor() {
    this.name = "BackgroundService";
    this.eventEmitter = new EventEmitter();
  }
  async initialize() {
    chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
      if (message.type === "PING") {
        sendResponse({ type: "PONG" });
        return true;
      }
      if (message.type === "FETCH_DATA") {
        sendResponse({
          success: true,
          data: {
            ...message.payload,
            timestamp: Date.now()
          }
        });
        return true;
      }
      if (message.type === "GET_AUTH_STATE") {
        const natsAuth2 = globalThis.contextManager.getAuthService();
        sendResponse({
          isAuthenticated: natsAuth2.state !== "disconnected",
          state: natsAuth2.state
        });
        return true;
      }
      if (message.type === "LOGOUT") {
        (async () => {
          try {
            const natsAuth2 = globalThis.contextManager.getAuthService();
            await natsAuth2.disconnect();
            chrome.runtime.sendMessage({
              type: "AUTH_STATE_CHANGED",
              isAuthenticated: false
            });
            sendResponse({ success: true });
          } catch (error) {
            sendResponse({
              success: false,
              error: error instanceof Error ? error.message : String(error)
            });
          }
        })();
        return true;
      }
    });
    const natsAuth = globalThis.contextManager.getAuthService();
    natsAuth.on("connected", () => {
      chrome.runtime.sendMessage({
        type: "AUTH_STATE_CHANGED",
        isAuthenticated: true
      });
    });
    natsAuth.on("disconnected", () => {
      chrome.runtime.sendMessage({
        type: "AUTH_STATE_CHANGED",
        isAuthenticated: false
      });
    });
  }
  async keepAlive() {
    chrome.alarms.create("keepAlive", { periodInMinutes: 1 });
    chrome.alarms.onAlarm.addListener((alarm) => {
      if (alarm.name === "keepAlive") {
        console.log("[Background] Service worker kept alive by alarm");
      }
    });
    const keepAliveInterval = 1e3 * 60 * 60;
    try {
      if (chrome.offscreen) {
        if (await chrome.offscreen.hasDocument()) {
          await chrome.offscreen.closeDocument();
        }
        async function startOffscreenDocument() {
          await chrome.offscreen.createDocument({
            url: "options.html",
            reasons: [chrome.offscreen.Reason.DOM_SCRAPING],
            justification: "Background processing of DOM APIs"
          });
        }
        setInterval(async () => {
          if (await chrome.offscreen.hasDocument()) {
            console.log("[Background] Offscreen document is still open");
          } else {
            startOffscreenDocument();
          }
        }, keepAliveInterval);
      }
    } catch (error) {
      console.error("[Background] Error setting up offscreen document:", error);
    }
  }
  async teardown() {
  }
}
function initializeBackground() {
  injectTestingHelpers({
    BackgroundService,
    NatsAuthService,
    RPCManager,
    Browser,
    Tabs,
    Session,
    Network,
    Page
  });
  const contextManager = createContextManager();
  const natsAuthService = new NatsAuthService();
  const rpcManager = new RPCManager();
  const browser = new Browser();
  const tabs = new Tabs();
  const session = new Session();
  const page = new Page();
  const network = new Network();
  contextManager.registerAuth(natsAuthService);
  rpcManager.registerAdapter(natsAuthService);
  const services = [browser, tabs, session, page];
  services.forEach((service) => {
    rpcManager.register(service);
  });
  contextManager.register(network);
  contextManager.register(rpcManager);
  const backgroundService = new BackgroundService();
  contextManager.register(backgroundService);
  contextManager.initialize().then(() => {
    globalThis.contextManager = contextManager;
  }).catch((error) => {
  });
}
initializeBackground();
chrome.runtime.onInstalled.addListener((details) => {
  if (details.reason === "install") {
    chrome.tabs.create({ url: "https://herd.garden/setup", active: true });
  }
});
//# sourceMappingURL=background.ts-o2h5qBav.js.map
